<?php
/***********************************************************************************
 * The contents of this file are subject to the Extension License Agreement
 * ("Agreement") which can be viewed at
 * https://www.espocrm.com/extension-license-agreement/.
 * By copying, installing downloading, or using this file, You have unconditionally
 * agreed to the terms and conditions of the Agreement, and You may not use this
 * file except in compliance with the Agreement. Under the terms of the Agreement,
 * You shall not license, sublicense, sell, resell, rent, lease, lend, distribute,
 * redistribute, market, publish, commercialize, or otherwise transfer rights or
 * usage to the software or any modified version or derivative work of the software
 * created by or for you.
 *
 * Copyright (C) 2024-2025 Letrium Ltd.
 *
 * License ID: f27e70ce6801a13265271f5669c8bc5c
 ************************************************************************************/

namespace Espo\Modules\Project\Tools\Project\Gantt;

use Espo\Core\Exceptions\BadRequest;
use Espo\Core\Exceptions\Forbidden;
use Espo\Core\Exceptions\NotFound;
use Espo\Core\Record\Collection as RecordCollection;
use Espo\Core\Record\ServiceContainer;
use Espo\Core\Select\SelectBuilderFactory;
use Espo\Modules\Project\Entities\Project;
use Espo\Modules\Project\Entities\ProjectGroup;
use Espo\Modules\Project\Entities\ProjectTask;
use Espo\Modules\Project\Tools\Project\PlanData;
use Espo\Modules\Project\Tools\ProjectTask\FromTasksLoader;
use Espo\ORM\Collection;
use Espo\ORM\EntityManager;
use Espo\ORM\Query\Select;
use RuntimeException;

class Service
{
    private const MAX_SIZE = 1000;

    /** @var string[] */
    private array $select = [
        'id',
        'name',
        'dateStart',
        'dateEnd',
        'status',
        'columnId',
        'columnName',
        'assignedUserId',
        'assignedUserName',
        'ownerId',
        'ownerName',
        'type',
        'order',
    ];

    public function __construct(
        private SelectBuilderFactory $selectBuilderFactory,
        private ServiceContainer $serviceContainer,
        private EntityManager $entityManager,
        private FromTasksLoader $fromTasksLoader,
    ) {}

    /**
     * @param string[] $collapsedGroupIds
     */
    public function get(Project $project, array $collapsedGroupIds): PlanData
    {
        $list = [];

        foreach ($this->getGroups($project) as $group) {
            $this->serviceContainer->getByClass(ProjectGroup::class)->prepareEntityForOutput($group);

            $tasks = !in_array($group->getId(), $collapsedGroupIds) ?
                $this->getTasks($project, $group) :
                null;

            $list[] = [$group, $tasks, null];
        }

        return new PlanData(
            groups: $list,
            total: $this->getTotal($project),
        );
    }

    /**
     * @return Collection<ProjectTask>
     */
    private function getTasks(Project $project, ProjectGroup $group): Collection
    {
        $query = $this->prepareQuery($project, $group);

        $tasks = $this->entityManager->getRDBRepositoryByClass(ProjectTask::class)->clone($query)->find();

        $this->fromTasksLoader->load($tasks, $query);

        $service = $this->serviceContainer->getByClass(ProjectTask::class);

        foreach ($tasks as $task) {
            $service->prepareEntityForOutput($task);
        }

        return $tasks;
    }

    /**
     * @return iterable<ProjectGroup>
     */
    private function getGroups(Project $project): iterable
    {
        /** @var Collection<ProjectGroup> */
        return $this->entityManager
            ->getRelation($project, 'groups')
            ->order('order')
            ->find();
    }

    private function prepareQuery(Project $project, ?ProjectGroup $group = null): Select
    {
        try {
            $builder = $this->selectBuilderFactory
                ->create()
                ->from(ProjectTask::ENTITY_TYPE)
                ->withStrictAccessControl()
                ->buildQueryBuilder();
        }
        catch (BadRequest|Forbidden $e) {
            throw new RuntimeException($e->getMessage(), 0, $e);
        }

        if ($group) {
            $builder->where(['groupId' => $group->getId()]);

            $builder->limit(0, self::MAX_SIZE);
        }

        return $builder
            ->select($this->select)
            ->order([])
            ->order('dateStart')
            ->order('dateEnd')
            ->order('order')
            ->where([
                'projectId' => $project->getId(),
                'parentTaskId' => null,
                'OR' => [
                    [
                        'type' => ProjectTask::TYPE_TASK,
                        'dateEnd!=' => null,
                    ],
                    [
                        'type' => ProjectTask::TYPE_MILESTONE,
                        'dateStart!=' => null,
                    ],
                ]
            ])
            ->build();
    }

    private function getTotal(Project $project): int
    {
        $query = $this->prepareQuery($project);

        return $this->entityManager
            ->getRDBRepositoryByClass(ProjectTask::class)
            ->clone($query)
            ->count();
    }

    /**
     * @return RecordCollection<ProjectTask>
     * @throws NotFound
     */
    public function getGroupTasks(ProjectGroup $group): RecordCollection
    {
        $project = $this->entityManager
            ->getRDBRepositoryByClass(Project::class)
            ->getById($group->getProjectId());

        if (!$project) {
            throw new NotFound("Project not found.");
        }

        $tasks = $this->getTasks($project, $group);

        return RecordCollection::create($tasks, RecordCollection::TOTAL_HAS_NO_MORE);
    }
}
