<?php
/***********************************************************************************
 * The contents of this file are subject to the Extension License Agreement
 * ("Agreement") which can be viewed at
 * https://www.espocrm.com/extension-license-agreement/.
 * By copying, installing downloading, or using this file, You have unconditionally
 * agreed to the terms and conditions of the Agreement, and You may not use this
 * file except in compliance with the Agreement. Under the terms of the Agreement,
 * You shall not license, sublicense, sell, resell, rent, lease, lend, distribute,
 * redistribute, market, publish, commercialize, or otherwise transfer rights or
 * usage to the software or any modified version or derivative work of the software
 * created by or for you.
 *
 * Copyright (C) 2024-2025 Letrium Ltd.
 *
 * License ID: f27e70ce6801a13265271f5669c8bc5c
 ************************************************************************************/

namespace Espo\Modules\Project\Tools\Project;

use Espo\Entities\User;
use Espo\Modules\Project\Entities\Project;
use Espo\Modules\Project\Entities\ProjectRole;
use Espo\ORM\EntityManager;

class MemberRoleProvider
{
    /** @var array<string, ?MemberRole> */
    private array $cache = [];

    public function __construct(
        private EntityManager $entityManager,
    ) {}

    public function get(User $user, string $projectId): ?MemberRole
    {
        $cacheKey = $user->getId() . '-' . $projectId;

        if (!array_key_exists($cacheKey, $this->cache)) {
            $this->cache[$cacheKey] = $this->getInternal($user, $projectId);
        }

        return $this->cache[$cacheKey];
    }

    private function getInternal(User $user, string $projectId): ?MemberRole
    {
        $projectUser = $this->entityManager
            ->getRDBRepository(Project::RELATIONSHIP_PROJECT_USER)
            ->where([
                'projectId' => $projectId,
                'userId' => $user->getId(),
            ])
            ->findOne();

        if (!$projectUser) {
            return null;
        }

        $role = $projectUser->get('role');
        $roleId = $projectUser->get('roleId');

        $roleEntity = null;

        if ($roleId) {
            $roleEntity = $this->entityManager->getRDBRepositoryByClass(ProjectRole::class)->getById($roleId);
        }

        if (!$roleEntity) {
            $roleEntity = $this->entityManager->getRDBRepositoryByClass(ProjectRole::class)->getNew();

            if ($role === Project::ROLE_OWNER || $role === Project::ROLE_EDITOR) {
                $roleEntity->populateMaxPermissions();
            } else {
                $roleEntity->populateDefaultMemberPermissions();
            }
        }

        return new MemberRole($role, $roleEntity);
    }
}
