<?php
/***********************************************************************************
 * The contents of this file are subject to the Extension License Agreement
 * ("Agreement") which can be viewed at
 * https://www.espocrm.com/extension-license-agreement/.
 * By copying, installing downloading, or using this file, You have unconditionally
 * agreed to the terms and conditions of the Agreement, and You may not use this
 * file except in compliance with the Agreement. Under the terms of the Agreement,
 * You shall not license, sublicense, sell, resell, rent, lease, lend, distribute,
 * redistribute, market, publish, commercialize, or otherwise transfer rights or
 * usage to the software or any modified version or derivative work of the software
 * created by or for you.
 *
 * Copyright (C) 2015-2025 Letrium Ltd.
 *
 * License ID: a67be8f75bc1974a987315ddc1024a78
 ************************************************************************************/

namespace Espo\Modules\Sales\Tools\Invoice;

use Espo\Core\Exceptions\Error;
use Espo\Core\Exceptions\Forbidden;
use Espo\Core\Exceptions\NotFound;
use Espo\Entities\Attachment;
use Espo\Entities\Email;
use Espo\Modules\Sales\Entities\Invoice;
use Espo\Modules\Sales\Tools\Invoice\EInvoice\UblGenerator;
use Espo\Modules\Sales\Tools\Quote\Email\GetAttributesParams;
use Espo\Modules\Sales\Tools\Quote\EmailService as BaseEmailService;
use Espo\ORM\EntityManager;

class EmailService
{
    public function __construct(
        private BaseEmailService $baseEmailService,
        private UblGenerator $ublGenerator,
        private EntityManager $entityManager
    ) {}

    /**
     * @throws Forbidden
     * @throws Error
     * @throws NotFound
     * @return array<string, mixed>
     */
    public function getAttributes(
        string $sourceId,
        string $templateId,
        ?GetAttributesParams $params = null,
        ?string $format = null,
    ): array {

        $attributes = $this->baseEmailService->getAttributes(Invoice::ENTITY_TYPE, $sourceId, $templateId, $params);

        if ($format) {
            $attributes = $this->addEInvoice($sourceId, $format, $attributes);
        }

        return $attributes;
    }

    /**
     * @param array<string, mixed> $attributes
     * @return array<string, mixed>
     * @throws Error
     * @throws NotFound
     */
    private function addEInvoice(string $sourceId, string $format, array $attributes): array
    {
        $invoice = $this->entityManager->getRDBRepositoryByClass(Invoice::class)->getById($sourceId);

        if (!$invoice) {
            throw new NotFound();
        }

        try {
            $attachment = $this->ublGenerator->generateAttachment($invoice, $format);
        } catch (EInvoice\Exceptions\RuleValidationFailure $e) {
            throw Error::createWithBody(
                $e->getMessage(),
                Error\Body::create()
                    ->withMessageTranslation('ublRuleValidationFailure', Invoice::ENTITY_TYPE, [
                        'ruleId' => $e->getRuleId(),
                        'message' => $e->getMessage(),
                    ])
            );
        }
        catch (EInvoice\Exceptions\UnknownFormat) {
            throw new Error("Unknown format.");
        }

        $attachment
            ->setType(Attachment::ROLE_ATTACHMENT)
            ->setTargetField('attachments')
            ->set('parentType', Email::ENTITY_TYPE);

        $this->entityManager->saveEntity($attachment);

        $attributes['attachmentsIds'] ??= [];
        $attributes['attachmentsNames'] ??= (object)[];

        $attributes['attachmentsIds'][] = $attachment->getId();
        $attributes['attachmentsNames']->{$attachment->getId()} = $attachment->getName();

        return $attributes;
    }
}
